@extends('Template::layouts.design')
@section('content')
    <form action="{{ route('product.buy.now') }}" method="post" class="position-relative" id="printArea">
        @csrf
        <input type="hidden" name="product_id" value="{{ $product->id }}">
        <input type="hidden" name="variant_id" value="{{ $productVariant?->id }}">
        <input type="hidden" class="d-none" value="{{ $cartProduct->quantity ?? 1 }}" name="quantity">
        <div id="printAreaSection" class="printarea-full-screen">
            @include($activeTemplate . '.partials.design_print_area')
        </div>
    </form>
    @include($activeTemplate . '.partials.product_drawing_modal')

    {{-- instruction modal --}}
    <div class="modal custom--modal" id="instructionModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h3 class="modal-title">@lang('Design Instruction')</h3>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    @php
                        $designInstructions = $product->design_instruction ?? ($product?->vendor?->design_instruction ?? []);
                    @endphp
                    <h4 class="mb-2"> @lang('Basic Instructions')</h4>
                    <div class="product-basic-design">
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fa-solid fa-ruler-combined"></i>
                            </span>
                            <span class="product-basic-design__info">@lang('Please make sure all your designs (text, images, shapes) stay strictly inside the marked design area (marked box).')</span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fa-solid fa-triangle-exclamation"></i>
                            </span>
                            <span class="product-basic-design__info"> @lang('Any design placed outside this area will not be printed.')</span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fa-solid fa-crosshairs"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('If your design area is small, you can zoom in using the zoom buttons to design with more accuracy.')
                            </span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fa-solid fa-object-ungroup"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('Make sure all elements (text or graphics) are fully visible and not cut off.')
                            </span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fa-solid fa-hand-pointer"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('Move On/Off: When the Move is OFF (⏸️), you can freely edit. When the Move is ON (🔛), editing is disabled — turn Move OFF to continue editing.')
                            </span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fas fa-text-width"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('Text Width: When you add text, leave some extra width on both sides. If there’s extra width beyond the visible text area, it should be trimmed or removed.')
                            </span>
                        </p>
                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fas fa-search-plus"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('Zoom In/Out: You can zoom in or out on the image area using the mouse wheel.')
                            </span>
                        </p>

                        <p class="product-basic-design__item">
                            <span class="product-basic-design__icon">
                                <i class="fas fa-arrows-alt"></i>
                            </span>
                            <span class="product-basic-design__info">
                                @lang('Move Image: Press and hold the Spacebar to move the image. You can also use the Move button to toggle move mode on or off.')
                            </span>
                        </p>
                    </div>
                    @if (!blank($designInstructions))
                        <div class="mt-3">
                            <h4 class="mb-0">@lang('Instructions From Vendor')</h4>
                            @foreach ($designInstructions as $designInstruction)
                                <p>{{ __($designInstruction) }}</p>
                            @endforeach
                        </div>
                    @endif
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn--dark btn--xs" data-bs-dismiss="modal">@lang('Close')</button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('script-lib')
    <script src="{{ asset('assets/global/js/fabric.min.js') }}"></script>
@endpush

@push('script')
    <script>
        (function($) {
            "use strict";

            const canvasMap = new Map();
            const desingAraBackground = "{{ $colorCode ? '#' . $colorCode : 'transparent' }}";

            function initDesignCanvas(container) {
                const canvasEl = container.find('canvas')[0];

                const hiddenInput = container.find('input[name="selected_area[]"]')[0];
                const imageURL = canvasEl.dataset.src;
                const editableAreaData = JSON.parse(canvasEl.dataset.selected || 'null');

                const canvas = new fabric.Canvas(canvasEl);
                let curvedMode = false;
                canvas.skipOffscreen = true;
                canvasMap.set(container[0], {
                    canvas,
                    container
                });

                const newHeight = container.find('.canvas-layout').height();
                const newWidth = container.find('.canvas-layout').width();

                container.find('.canvas-container').css({
                    height: newHeight + 'px',
                    width: newWidth + 'px'
                });

                container.find('.upper-canvas').css({
                    width: '100%',
                    height: '100%'
                });

                canvasEl.style.width = '100%';
                canvasEl.style.height = '100%';

                fabric.Object.prototype.toObject = (function(toObject) {
                    return function(additionalProps) {
                        return toObject.call(this, ['shapeName', 'originalSrc', 'customInfo']
                            .concat(additionalProps || []));
                    };
                })(fabric.Object.prototype.toObject);

                function restrictObject(obj) {
                    if (!editableAreaData) return;

                    const bgImg = canvas.backgroundImage;
                    if (!bgImg) return;

                    const imgScale = bgImg.scaleX; // assume uniform scaling
                    const imgLeft = bgImg.left;
                    const imgTop = bgImg.top;

                    const scaledArea = {
                        left: imgLeft + editableAreaData.left * imgScale,
                        top: imgTop + editableAreaData.top * imgScale,
                        width: editableAreaData.width * imgScale,
                        height: editableAreaData.height * imgScale
                    };

                    const maxRight = scaledArea.left + scaledArea.width;
                    const maxBottom = scaledArea.top + scaledArea.height;

                    // Movement restriction
                    obj.on('moving', () => {
                        const objW = obj.getScaledWidth();
                        const objH = obj.getScaledHeight();

                        let leftEdge, rightEdge, topEdge, bottomEdge;

                        if (obj.originX === 'center') {
                            leftEdge = obj.left - objW / 2;
                            rightEdge = obj.left + objW / 2;
                        } else { // 'left'
                            leftEdge = obj.left;
                            rightEdge = obj.left + objW;
                        }

                        if (obj.originY === 'center') {
                            topEdge = obj.top - objH / 2;
                            bottomEdge = obj.top + objH / 2;
                        } else { // 'top'
                            topEdge = obj.top;
                            bottomEdge = obj.top + objH;
                        }

                        // Restrict movement
                        if (leftEdge < scaledArea.left) obj.left += scaledArea.left - leftEdge;
                        if (topEdge < scaledArea.top) obj.top += scaledArea.top - topEdge;
                        if (rightEdge > maxRight) obj.left -= rightEdge - maxRight;
                        if (bottomEdge > maxBottom) obj.top -= bottomEdge - maxBottom;
                    });

                    // Scaling restriction
                    obj.on('scaling', () => {
                        let objW = obj.width * obj.scaleX;
                        let objH = obj.height * obj.scaleY;

                        const maxW = scaledArea.width;
                        const maxH = scaledArea.height;

                        if (objW > maxW) {
                            obj.scaleX = maxW / obj.width;
                            objW = obj.width * obj.scaleX;
                        }
                        if (objH > maxH) {
                            obj.scaleY = maxH / obj.height;
                            objH = obj.height * obj.scaleY;
                        }

                        if (obj.left < scaledArea.left) obj.left = scaledArea.left;
                        if (obj.top < scaledArea.top) obj.top = scaledArea.top;
                        if (obj.left + obj.getScaledWidth() > maxRight) obj.left = maxRight - obj
                            .getScaledWidth();
                        if (obj.top + obj.getScaledHeight() > maxBottom) obj.top = maxBottom - obj
                            .getScaledHeight();
                    });
                }

                function drawEditableArea(isSvg = false) {
                    if (!editableAreaData) return;

                    const bgImg = canvas.backgroundImage;
                    if (!bgImg) return;

                    // Image scale and offset
                    const imgScale = bgImg.scaleX; // uniform scale
                    const imgLeft = bgImg.left;
                    const imgTop = bgImg.top;
                    const strokeColor = pickBorderColor(desingAraBackground);
                    const areaFill = isSvg ? "transparent" : desingAraBackground;

                    let area;

                    if (editableAreaData.type === 'circle') {
                        // Circle: calculate center
                        const radius = editableAreaData.radius * imgScale;
                        const centerX = imgLeft + editableAreaData.left * imgScale + radius;
                        const centerY = imgTop + editableAreaData.top * imgScale + radius;

                        area = new fabric.Circle({
                            left: centerX,
                            top: centerY,
                            originX: 'center',
                            originY: 'center',
                            radius: radius,
                            fill: areaFill,
                            stroke: strokeColor,
                            strokeDashArray: [5, 5],
                            selectable: false,
                            evented: false,
                            objectCaching: false,
                            isEditableBorder: true
                        });

                    } else {
                        // Default rectangle
                        area = new fabric.Rect({
                            left: imgLeft + editableAreaData.left * imgScale,
                            top: imgTop + editableAreaData.top * imgScale,
                            width: editableAreaData.width * imgScale,
                            height: editableAreaData.height * imgScale,
                            fill: areaFill,
                            stroke: strokeColor,
                            strokeDashArray: [5, 5],
                            selectable: false,
                            evented: false,
                            objectCaching: false,
                            isEditableBorder: true
                        });
                    }

                    canvas.add(area);
                    canvas.sendToBack(area);
                }

                function saveJson() {
                    const objectsToSave = canvas.getObjects().filter(obj => !obj.isEditableBorder && !obj
                        .isBackgroundImage);
                    const json = canvas.toJSON(['selectable', 'angle', 'shapeName', 'customInfo']);
                    json.objects = objectsToSave.map(obj => obj.toObject(['selectable', 'angle', 'shapeName',
                        'customInfo'
                    ]));
                    hiddenInput.value = JSON.stringify(json);
                }

                // load image and old drawing
                const containerEl = container.find('.canvas-layout')[0];
                const canvasWidth = containerEl.clientWidth;
                const canvasHeight = containerEl.clientHeight;
                canvas.setWidth(canvasWidth);
                canvas.setHeight(canvasHeight);

                // Set default background image
                fabric.Image.fromURL(imageURL, img => {
                    if (imageURL.endsWith('.svg')) {
                        fetch(imageURL)
                            .then(res => res.text())
                            .then(svgText => {
                                const parser = new DOMParser();
                                const svgDoc = parser.parseFromString(svgText, "image/svg+xml");
                                const svgEl = svgDoc.documentElement;

                                const newFillColor = desingAraBackground; // your new fill color
                                svgEl.querySelectorAll(':scope > path').forEach(el => {
                                    el.setAttribute('fill', newFillColor);
                                });

                                const serializer = new XMLSerializer();
                                const svgStr = serializer.serializeToString(svgEl);
                                const svgDataUrl = "data:image/svg+xml;base64," + btoa(unescape(
                                    encodeURIComponent(svgStr)));

                                loadFabricImage(svgDataUrl, true);
                            });
                    } else {
                        loadFabricImage(imageURL); // raster images
                    }

                    function loadFabricImage(url, isSvg = false) {
                        fabric.Image.fromURL(url, imgInstance => {
                            imgInstance.set({
                                selectable: false
                            });

                            const scaleX = canvasWidth / imgInstance.width;
                            const scaleY = canvasHeight / imgInstance.height;
                            const scale = Math.min(scaleX, scaleY);

                            imgInstance.scale(scale);
                            imgInstance.left = (canvasWidth - imgInstance.width * scale) / 2;
                            imgInstance.top = (canvasHeight - imgInstance.height * scale) / 2;

                            canvas.setBackgroundImage(imgInstance, canvas.renderAll.bind(canvas), {
                                originX: 'left',
                                originY: 'top'
                            });

                            drawEditableArea(isSvg);
                            canvas.requestRenderAll();
                        });
                    }
                });

                // Load saved objects
                if (hiddenInput.value) {
                    try {
                        const savedJson = JSON.parse(hiddenInput.value);
                        canvas.loadFromJSON(savedJson, () => {
                            canvas.getObjects().forEach(obj => restrictObject(obj));
                            canvas.requestRenderAll();
                        });
                    } catch (e) {}
                } else {
                    canvas.requestRenderAll();
                }

                // Event bindings
                container.find('[data-action]').each((i, btn) => {
                    const action = btn.dataset.action;
                    btn.addEventListener('click', () => {
                        const handler = window[action];
                        if (typeof handler === 'function') handler(canvas, editableAreaData);
                        saveJson();
                    });
                });

                container.find('[data-action="zoomIn"]').on('click', () => zoom(canvas, +0.1, container));
                container.find('[data-action="zoomOut"]').on('click', () => zoom(canvas, -0.1, container));
                container.find('.zoom-reset').on('click', () => zoomReset(canvas, container));
                container.find('[data-action="toggleMove"]').on('click', () => toggleMove(canvas, container));

                container.find('[data-action="addText"]').on('click', () =>
                    addText(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="fontSize"]').on('change input', () => fontSize(canvas, container));
                container.find('[data-action="fontFamily"]').on('change', () => fontFamily(canvas, container));
                container.find('[data-action="addRect"]').on('click', () =>
                    addRect(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="addCircle"]').on('click', () =>
                    addCircle(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="addTriangle"]').on('click', () =>
                    addTriangle(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="addLine"]').on('click', () =>
                    addLine(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="addStar"]').on('click', () =>
                    addStar(canvas, editableAreaData, restrictObject, container));
                container.find('[data-action="addPolygon"]').on('click', () =>
                    addPolygon(canvas, editableAreaData, restrictObject, container));

                container.find('[data-action="addHeart"]').on('click', () => addHeart(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addDiamond"]').on('click', () => addDiamond(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addPentagon"]').on('click', () => addPentagon(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addHexagon"]').on('click', () => addHexagon(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addCloud"]').on('click', () => addCloud(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addArrow"]').on('click', () => addArrow(canvas, editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addSpeechBubble"]').on('click', () => addSpeechBubble(canvas,
                    editableAreaData,
                    restrictObject, container));
                container.find('[data-action="addMoon"]').on('click', () => addMoon(canvas, editableAreaData,
                    restrictObject, container));

                container.find('[data-action="toggleBold"]').on('click', () => toggleBold(canvas));
                container.find('[data-action="toggleUnderline"]').on('click', () => toggleUnderline(canvas));
                container.find('[data-action="deleteSelected"]').on('click', () => deleteSelected(canvas));
                container.find('[data-action="clearDesign"]').on('click', () => clearDesign(canvas));
                container.find('[data-action="colorPicker"]').on('change input', () => colorPicker(canvas, container));
                container.find('[data-action="shapeColorPicker"]').on('change input', () => shapeColorPicker(canvas,
                    container));
                container.find('[data-action="toggleTextCurbe"]').on('click', () => toggleTextCurbe(canvas,
                    container));
                container.find('[data-action="toggleItalic"]').on('click', () => toggleItalic(canvas,
                    container));
                container.find('[data-action="fullScreen"]').on('click', () => toggleFullScreen($(
                    '#printAreaSection'), canvas, container, editableAreaData));
                container.find('[data-action="draw"]').on('click', () => enableDrawModal(canvas, container,
                    editableAreaData, restrictObject));
                container.find('[data-action="fitScreen"]').on('click', () => fitScreen(canvas, container));


                // Trigger file selector on button click
                container.find('[data-action="uploadImage"]').on('click', () => {
                    container.find('[data-action="uploadImageInput"]').on('click');
                });

                // Handle image file selection and add image to canvas
                container.find('[data-action="uploadImageInput"]').on('change', function(event) {
                    const file = event.target.files[0];
                    if (file && file.type.startsWith('image/')) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            fabric.Image.fromURL(e.target.result, img => {
                                const padding = 10;
                                // Get scaled editable area (same as text)
                                const bgImg = canvas.backgroundImage;
                                if (!bgImg) return;
                                const imgScale = bgImg.scaleX; // assume uniform scale
                                const imgLeft = bgImg.left;
                                const imgTop = bgImg.top;
                                const scaledArea = {
                                    left: imgLeft + editableAreaData.left * imgScale,
                                    top: imgTop + editableAreaData.top * imgScale,
                                    width: editableAreaData.width * imgScale,
                                    height: editableAreaData.height * imgScale
                                };
                                const maxWidth = scaledArea.width - padding * 2;
                                const maxHeight = scaledArea.height - padding * 2;
                                img.set({
                                    left: scaledArea.left + padding,
                                    top: scaledArea.top + padding,
                                    selectable: true,
                                    hasControls: true,
                                    lockRotation: false,
                                    lockScalingFlip: true,
                                    originX: 'left',
                                    originY: 'top',
                                    shapeName: "Image"
                                });
                                const scaleX = maxWidth / img.width;
                                const scaleY = maxHeight / img.height;
                                const scale = Math.min(scaleX, scaleY, 1);
                                img.originalSrc = e.target.result;
                                img.scale(scale);
                                restrictObject(img);
                                // Force inside area immediately
                                const maxRight = scaledArea.left + scaledArea.width;
                                const maxBottom = scaledArea.top + scaledArea.height;
                                if (img.left < scaledArea.left) img.left = scaledArea.left;
                                if (img.top < scaledArea.top) img.top = scaledArea.top;
                                if (img.left + img.getScaledWidth() > maxRight) img.left =
                                    maxRight - img.getScaledWidth();
                                if (img.top + img.getScaledHeight() > maxBottom) img.top =
                                    maxBottom - img.getScaledHeight();
                                canvas.add(img).setActiveObject(img);
                                canvas.requestRenderAll();
                                saveJson();
                                event.target.value = '';
                            });
                        };
                        reader.readAsDataURL(file);
                    } else {
                        alert(`@lang('Please select a valid image file')`);
                    }
                });

                enableMouseWheelZoom(canvas, container);

                // Save on changes
                canvas.on('object:modified', saveJson);
                canvas.on('object:added', saveJson);
                canvas.on('object:removed', saveJson);
                canvas.on('mouse:up', saveJson);

                // Move mode (pan)
                canvas.on('mouse:down', function(opt) {
                    if (!canvas.isMoveMode) return;
                    const evt = opt.e;
                    canvas.lastPosX = evt.clientX || (evt.touches && evt.touches[0] && evt.touches[0].clientX);
                    canvas.lastPosY = evt.clientY || (evt.touches && evt.touches[0] && evt.touches[0].clientY);
                    canvas.isDragging = true;
                    canvas.selection = false;
                });

                canvas.on('mouse:move', function(opt) {
                    if (!canvas.isDragging || !canvas.isMoveMode) return;
                    const e = opt.e;
                    const currentX = e.clientX || (e.touches && e.touches[0] && e.touches[0].clientX);
                    const currentY = e.clientY || (e.touches && e.touches[0] && e.touches[0].clientY);

                    if (currentX !== undefined && currentY !== undefined) {
                        const vpt = canvas.viewportTransform;
                        vpt[4] += currentX - canvas.lastPosX;
                        vpt[5] += currentY - canvas.lastPosY;
                        canvas.requestRenderAll();
                        canvas.lastPosX = currentX;
                        canvas.lastPosY = currentY;
                    }
                });

                canvas.on('mouse:up', function() {
                    canvas.isDragging = false;
                    if (!canvas.isMoveMode) canvas.selection = true;
                });


                // TODO:
                canvas.on('selection:created', handleSelection);
                canvas.on('selection:updated', handleSelection);

                function handleSelection(e) {
                    if (e.selected && e.selected.length > 0) {
                        const obj = e.selected[0]; // first selected object
                        if (!obj) return;

                        if (obj.type == 'textbox') {
                            openDrawer('textbox');
                        } else if (["rect", "circle", "triangle", "line", "polygon", "path", "group"].includes(obj.type)) {
                            openDrawer('shape');
                        } else if (obj.type == 'image') {
                            openDrawer('draw');
                        }
                    }
                }

                fitScreen(canvas, container);
                // Track hover so spacebar only affects this canvas
                container.on('mouseenter touchstart', () => {
                    hoveredCanvasInfo = {
                        canvas,
                        container
                    };
                });

                container.on('mouseleave', () => {
                    // If leaving and move mode is on, turn it off
                    if (hoveredCanvasInfo && hoveredCanvasInfo.canvas === canvas) {
                        if (canvas.isMoveMode) {
                            toggleMove(canvas, container);
                        }
                        hoveredCanvasInfo = null;
                    }
                });

                canvas.upperCanvasEl.addEventListener('touchstart', function(e) {
                    if (!canvas.isMoveMode || e.touches.length !== 1) return;

                    e.preventDefault();
                    const touch = e.touches[0];
                    canvas.lastPosX = touch.clientX;
                    canvas.lastPosY = touch.clientY;
                    canvas.isDragging = true;
                    canvas.selection = false;
                });

                canvas.upperCanvasEl.addEventListener('touchmove', function(e) {
                    if (!canvas.isDragging || !canvas.isMoveMode || e.touches.length !== 1) return;

                    e.preventDefault();
                    const touch = e.touches[0];
                    const vpt = canvas.viewportTransform;
                    vpt[4] += touch.clientX - canvas.lastPosX;
                    vpt[5] += touch.clientY - canvas.lastPosY;
                    canvas.requestRenderAll();
                    canvas.lastPosX = touch.clientX;
                    canvas.lastPosY = touch.clientY;
                });

                canvas.upperCanvasEl.addEventListener('touchend', function(e) {
                    if (e.touches.length === 0) {
                        canvas.isDragging = false;
                        if (!canvas.isMoveMode) canvas.selection = true;
                    }
                });

                container.on('mouseleave', () => {
                    // If leaving and move mode is on, turn it off
                    if (hoveredCanvasInfo && hoveredCanvasInfo.canvas === canvas) {
                        if (canvas.isMoveMode) {
                            toggleMove(canvas, container);
                        }
                        hoveredCanvasInfo = null;
                    }
                });

                // Touch-specific hover simulation for mobile devices
                container.on('touchend', (e) => {
                    // Delay clearing to allow for quick gestures
                    setTimeout(() => {
                        if (hoveredCanvasInfo && hoveredCanvasInfo.canvas === canvas) {
                            if (canvas.isMoveMode && !canvas.isDragging) {
                                toggleMove(canvas, container);
                            }
                            hoveredCanvasInfo = null;
                        }
                    }, 100);
                });

                document.addEventListener('keydown', function(e) {
                    // Check if Delete or Backspace is pressed
                    if (e.key === 'Delete') {
                        e.preventDefault();
                        deleteSelected(canvas);
                    }
                });
            }

            function pickBorderColor(areaFill) {
                // Transparent hole or no fill → keep red
                if (!areaFill || areaFill === 'transparent') return 'red';

                const normalized = areaFill.toLowerCase();
                if (normalized === '#ff0000' || normalized === 'red') {
                    // If same as red, choose black instead
                    return '#000000';
                }

                // Otherwise calculate contrast
                return getContrastingColor(areaFill);
            }

            function getContrastingColor(hex) {
                hex = hex.replace('#', '');
                if (hex.length === 3) {
                    hex = hex.split('').map(c => c + c).join('');
                }

                const r = parseInt(hex.substr(0, 2), 16);
                const g = parseInt(hex.substr(2, 2), 16);
                const b = parseInt(hex.substr(4, 2), 16);

                const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
                return luminance > 0.5 ? '#000000' : '#FFFFFF';
            }

            function zoom(canvas, step, container) {
                const z = Math.max(.1, Math.min(10, canvas.getZoom() + step));
                canvas.setZoom(z);

                const percentage = Math.round(z * 100) + '%';
                container.find('.zoom-reset').text(percentage);
            }

            function zoomReset(canvas, container) {
                canvas.setZoom(1);
                container.find('.zoom-reset').text('100%');
            }

            function toggleMove(canvas, container) {
                canvas.isMoveMode = !canvas.isMoveMode;
                canvas.selection = !canvas.isMoveMode;

                canvas.forEachObject(obj => {
                    if (obj.type !== 'image' && obj.evented !== false) {
                        obj.selectable = !canvas.isMoveMode;
                    }
                });

                canvas.discardActiveObject();
                canvas.requestRenderAll();

                const moveModeBtn = $(canvas.upperCanvasEl)
                    .closest('.printSideAreaItem')
                    .find('[data-action="toggleMove"]');
                moveModeBtn.toggleClass('active-move', canvas.isMoveMode);
                $(canvas.upperCanvasEl).css('cursor', canvas.isMoveMode ? 'move' : 'default');
                zoom(canvas, 0.0001, container)
            }

            $(document).ready(function() {
                // Initialize only the first active tab
                const firstTabContent = $('.tab-pane.active'); // Bootstrap adds 'active' class
                firstTabContent.find('.printSideAreaItem').each(function() {
                    initDesignCanvas($(this));
                });

                // Initialize other tabs on activation
                $('button[data-bs-toggle="pill"]').on('shown.bs.tab', function(e) {
                    const targetId = $(e.target).data('bs-target'); // e.g. #id1
                    const targetTab = $(targetId);

                    targetTab.find('.printSideAreaItem').each(function() {
                        const canvasEl = $(this).find('canvas')[0];

                        // Check if canvas already initialized (using your canvasMap)
                        if (!canvasMap.has(this)) {
                            initDesignCanvas($(this));
                        } else {
                            // If already initialized, just resize & render in case tab was hidden
                            const {
                                canvas,
                                container
                            } = canvasMap.get(this);
                            const containerEl = container.find('.canvas-layout')[0];
                            canvas.setWidth(containerEl.clientWidth);
                            canvas.setHeight(containerEl.clientHeight);
                            canvas.requestRenderAll();

                        }
                    });
                });
            });

            // add text
            function addText(canvas, editableAreaData, restrictObject, container) {
                const colorInput = container.find('.color-picker')[0];
                const fontSizeInput = container.find('.font-size')[0];
                const fontFamilySelect = container.find('.font-family')[0];

                const color = colorInput.value;
                const fontSize = parseInt(fontSizeInput.value);
                const fontFamily = fontFamilySelect.value;

                // Get scaled editable area
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX; // uniform scale
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const maxWidth = scaledArea.width - 10;
                const maxHeight = scaledArea.height - 10;

                const text = new fabric.Textbox('Text', {
                    left: scaledArea.left + 5,
                    top: scaledArea.top + 5,
                    width: maxWidth,
                    fontSize: fontSize,
                    fill: color,
                    fontFamily: fontFamily,
                    selectable: !canvas.isMoveMode,
                    editable: !canvas.isMoveMode,
                    hasControls: true,
                    lockScalingFlip: true,
                    lockRotation: false,
                });

                // Apply restriction for scaled area
                restrictObject(text);

                // Force initial position inside scaled area
                const maxRight = scaledArea.left + scaledArea.width;
                const maxBottom = scaledArea.top + scaledArea.height;

                if (text.left < scaledArea.left) text.left = scaledArea.left;
                if (text.top < scaledArea.top) text.top = scaledArea.top;
                if (text.left + text.getScaledWidth() > maxRight) text.left = maxRight - text.getScaledWidth();
                if (text.top + text.getScaledHeight() > maxBottom) text.top = maxBottom - text.getScaledHeight();

                canvas.add(text).setActiveObject(text);
                canvas.requestRenderAll();

                // Restrict moving/scaling inside scaled area
                text.on('moving', () => {
                    if (text.left < scaledArea.left) text.left = scaledArea.left;
                    if (text.top < scaledArea.top) text.top = scaledArea.top;
                    if (text.left + text.getScaledWidth() > maxRight) text.left = maxRight - text
                        .getScaledWidth();
                    if (text.top + text.getScaledHeight() > maxBottom) text.top = maxBottom - text
                        .getScaledHeight();
                });

                text.on('scaling', () => {
                    const newWidth = text.width * text.scaleX;
                    const newHeight = text.height * text.scaleY;

                    if (newWidth > maxWidth) text.scaleX = maxWidth / text.width;
                    if (newHeight > maxHeight) text.scaleY = maxHeight / text.height;

                    if (text.left < scaledArea.left) text.left = scaledArea.left;
                    if (text.top < scaledArea.top) text.top = scaledArea.top;
                    if (text.left + text.getScaledWidth() > maxRight) text.left = maxRight - text
                        .getScaledWidth();
                    if (text.top + text.getScaledHeight() > maxBottom) text.top = maxBottom - text
                        .getScaledHeight();
                });
            }

            // Change font size of selected text
            function fontSize(canvas, container) {
                const obj = canvas.getActiveObject();
                if (obj && obj.type === 'textbox') {
                    obj.set('fontSize', parseInt(container.find('.font-size')[0].value));
                    canvas.requestRenderAll();
                }
            }

            // Change color of selected text or shape fill color using single color picker
            function colorPicker(canvas, container) {
                const obj = canvas.getActiveObject();
                if (!obj) return;

                const color = container.find('.color-picker')[0].value;

                if (obj.type === 'textbox') {
                    obj.set('fill', color);
                } else if (obj.set) {
                    obj.set('fill', color);
                }
                canvas.requestRenderAll();
            }

            // Change color of selected text or shape fill color using single color picker
            function shapeColorPicker(canvas, container) {
                const obj = canvas.getActiveObject();
                if (!obj) return;

                const color = container.find('.shape-color-picker')[0].value;

                if (obj.type !== 'textbox') {
                    obj.set('fill', color);
                }
                canvas.requestRenderAll();
            }


            function toggleTextCurbe(canvas, container) {
                const obj = canvas.getActiveObject();
                if (!obj || obj.type !== 'textbox') {
                    alert("Select a textbox first!");
                    return;
                }

                if (!obj.curvedMode) {
                    createCurvedText(canvas, obj, 90);
                    obj.curvedMode = true;
                } else {
                    removeCurvedText(obj);
                    obj.curvedMode = false;
                }

                canvas.requestRenderAll();
            }

            function createCurvedText(canvas, textbox, curveDeg = 90) {
                if (textbox._curvedGroup) return; // Already curved

                const text = textbox.text || '';
                if (!text) return;

                const radius = 30; // Reduced radius for less width
                const degStep = curveDeg / Math.max(text.length - 1, 1);
                const startAngle = -curveDeg / 2;

                // Hide original textbox
                textbox.set({
                    visible: false
                });
                canvas.requestRenderAll();

                const chars = [];
                for (let i = 0; i < text.length; i++) {
                    const angle = startAngle + degStep * i;
                    const rad = angle * (Math.PI / 180);

                    const char = new fabric.Text(text[i], {
                        fontSize: textbox.fontSize,
                        fontFamily: textbox.fontFamily,
                        fontWeight: textbox.fontWeight,
                        fill: textbox.fill,
                        originX: 'center',
                        originY: 'center',
                        selectable: true,
                        left: radius * Math.cos(rad),
                        top: radius * Math.sin(rad),
                        angle: angle + 90,
                    });

                    chars.push(char);
                }

                const group = new fabric.Group(chars, {
                    left: textbox.left,
                    top: textbox.top,
                    originX: 'center',
                    originY: 'center',
                    selectable: true,
                    hasControls: true,
                    lockScalingFlip: true,
                });

                // Save original textbox link
                group._originalTextbox = textbox;
                textbox._curvedGroup = group;

                // Save info for admin view
                group.customInfo = {
                    text: textbox.text,
                    fontSize: textbox.fontSize,
                    fontFamily: textbox.fontFamily,
                    fontWeight: textbox.fontWeight,
                    fill: textbox.fill,
                    left: textbox.left,
                    top: textbox.top,
                    angle: textbox.angle || 0,
                };

                canvas.add(group).setActiveObject(group);
                canvas.requestRenderAll();
            }

            function removeCurvedText(textbox) {
                const group = textbox._curvedGroup;
                if (!group) return;

                const canvas = textbox.canvas;
                if (canvas) {
                    canvas.remove(group);
                    textbox.set({
                        visible: true
                    });
                    delete textbox._curvedGroup;
                    canvas.setActiveObject(textbox);
                    canvas.requestRenderAll();
                }
            }

            function toggleItalic(canvas, container) {
                const obj = canvas.getActiveObject();

                if (!obj || obj.type !== 'textbox') {
                    alert("Select a textbox first!");
                    return;
                }

                const currentStyle = obj.fontStyle || 'normal';
                obj.set('fontStyle', currentStyle === 'italic' ? 'normal' : 'italic');

                canvas.requestRenderAll();
            }

            // Change font family of selected text
            function fontFamily(canvas, container) {
                const obj = canvas.getActiveObject();
                if (obj && obj.type === 'textbox') {
                    obj.set('fontFamily', container.find('.font-family')[0].value);
                    canvas.requestRenderAll();
                }
            }

            // Add Rectangle
            function addRect(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                // Get scaled editable area (same as drawEditableArea)
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX; // assume uniform scaling
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const width = scaledArea.width - padding * 2;
                const height = scaledArea.height - padding * 2;

                const left = scaledArea.left + padding;
                const top = scaledArea.top + padding;

                const rect = new fabric.Rect({
                    left: left,
                    top: top,
                    width: width,
                    height: height,
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockScalingFlip: true,
                    lockRotation: false,
                    shapeName: "Rectangle"
                });

                // Apply restriction so it cannot move/scale outside red border
                restrictObject(rect);

                canvas.add(rect).setActiveObject(rect);
                canvas.requestRenderAll();
            }

            function addCircle(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                // Get scaled editable area (same as drawEditableArea)
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX; // assume uniform scaling
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const diameter = Math.min(scaledArea.width, scaledArea.height) - padding * 2;

                const left = scaledArea.left + padding;
                const top = scaledArea.top + padding;

                const circle = new fabric.Circle({
                    left: left,
                    top: top,
                    radius: diameter / 2,
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockScalingFlip: true,
                    lockRotation: false,
                    originX: "left", // same as rect
                    originY: "top", // same as rect
                    shapeName: "Circle"
                });

                // Apply restriction so it cannot move/scale outside red border
                restrictObject(circle);

                canvas.add(circle).setActiveObject(circle);
                canvas.requestRenderAll();
            }

            // Add Circle
            function addTriangle(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                // Get scaled editable area (same as drawEditableArea)
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX; // uniform scaling
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const maxWidth = scaledArea.width - padding * 2;
                const maxHeight = scaledArea.height - padding * 2;

                // 🔑 Important: anchor at top-left, same as rect & circle
                const left = scaledArea.left + padding;
                const top = scaledArea.top + padding;

                const triangle = new fabric.Triangle({
                    left: left,
                    top: top,
                    width: maxWidth,
                    height: maxHeight,
                    fill: color,
                    originX: 'left', // ✅ same as circle
                    originY: 'top', // ✅ same as circle
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Triangle"
                });

                // Restrict to stay inside editable area
                restrictObject(triangle);

                canvas.add(triangle).setActiveObject(triangle);
                canvas.requestRenderAll();
            }

            function addLine(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                // Get scaled editable area (same as drawEditableArea)
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX; // assume uniform scaling
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const startX = 0;
                const startY = 0;
                const endX = scaledArea.width - padding * 2;
                const endY = 0;

                const line = new fabric.Line([startX, startY, endX, endY], {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + scaledArea.height / 2, // put line in middle of editable area
                    stroke: color,
                    strokeWidth: 3,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    originX: 'left',
                    originY: 'center',
                    shapeName: "Line"
                });

                restrictObject(line);
                canvas.add(line).setActiveObject(line);
                canvas.requestRenderAll();
            }

            function addStar(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const points = 5;
                const padding = 10;

                // Get scaled editable area (same as drawEditableArea)
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const radius = Math.min(scaledArea.width, scaledArea.height) / 4;
                const angle = Math.PI / points;
                const path = [];

                for (let i = 0; i < 2 * points; i++) {
                    const r = (i % 2 === 0) ? radius : radius / 2;
                    const x = r * Math.sin(i * angle);
                    const y = -r * Math.cos(i * angle);
                    path.push({
                        x,
                        y
                    });
                }

                const star = new fabric.Polygon(path, {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    fill: color,
                    originX: 'left',
                    originY: 'top',
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Star"
                });

                restrictObject(star);
                canvas.add(star).setActiveObject(star);
                canvas.requestRenderAll();
            }

            function addPolygon(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const sides = 6;
                const padding = 10;

                // Get scaled editable area
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const radius = Math.min(scaledArea.width, scaledArea.height) / 4;
                const angle = 2 * Math.PI / sides;
                const path = [];

                for (let i = 0; i < sides; i++) {
                    const x = radius * Math.cos(i * angle);
                    const y = radius * Math.sin(i * angle);
                    path.push({
                        x,
                        y
                    });
                }

                const polygon = new fabric.Polygon(path, {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    fill: color,
                    originX: 'left',
                    originY: 'top',
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Polygon"
                });

                restrictObject(polygon);
                canvas.add(polygon).setActiveObject(polygon);
                canvas.requestRenderAll();
            }


            function toggleBold(canvas) {
                const obj = canvas.getActiveObject();
                if (obj && (obj.type === 'textbox' || obj.type === 'text')) {
                    obj.set('fontWeight', obj.fontWeight === 'bold' ? 'normal' : 'bold');
                    canvas.requestRenderAll();
                }
            }


            function toggleUnderline(canvas) {
                const obj = canvas.getActiveObject();
                if (obj && (obj.type === 'textbox' || obj.type === 'text')) {
                    obj.set('underline', !obj.underline);
                    canvas.requestRenderAll();
                }
            }


            // selected object delete
            function deleteSelected(canvas) {
                const obj = canvas.getActiveObject();
                if (obj && obj !== canvas.backgroundImage) {
                    canvas.remove(obj);
                }
            }

            // Clear all objects except background and editable area
            function clearDesign(canvas) {
                if (!window.confirm(`@lang('Are you sure you want to delete all objects?')`)) {
                    return;
                }


                const objs = canvas.getObjects().slice();
                objs.forEach(obj => {
                    if (obj !== canvas.backgroundImage && obj.selectable === true) {
                        canvas.remove(obj);
                    }
                });
            }


            // Heart
            function addHeart(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;

                const path = new fabric.Path(
                    'M 272 304 C 272 304 272 240 336 240 C 400 240 400 320 336 384 L 272 448 L 208 384 C 144 320 144 240 208 240 C 272 240 272 304 272 304 Z'
                );

                path.set({
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    scaleX: 0.1,
                    scaleY: 0.1,
                    shapeName: "Heart"
                });

                restrictObject(path);
                canvas.add(path).setActiveObject(path);
            }

            // Diamond
            function addDiamond(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const area = getScaledArea(canvas, editableAreaData);

                const size = Math.min(area.width, area.height) / 3;
                const points = [{
                        x: 0,
                        y: -size / 2
                    },
                    {
                        x: size / 2,
                        y: 0
                    },
                    {
                        x: 0,
                        y: size / 2
                    },
                    {
                        x: -size / 2,
                        y: 0
                    }
                ];

                const diamond = new fabric.Polygon(points, {
                    left: area.left + area.width / 2,
                    top: area.top + area.height / 2,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Diamond"
                });

                restrictObject(diamond);
                canvas.add(diamond).setActiveObject(diamond);
            }

            // Pentagon
            function addPentagon(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const sides = 5;
                const radius = Math.min(scaledArea.width, scaledArea.height) / 4;
                const angle = 2 * Math.PI / sides;
                const path = [];

                for (let i = 0; i < sides; i++) {
                    path.push({
                        x: radius + radius * Math.cos(i * angle),
                        y: radius + radius * Math.sin(i * angle)
                    });
                }

                const pentagon = new fabric.Polygon(path, {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Pentagon"
                });

                restrictObject(pentagon);
                canvas.add(pentagon).setActiveObject(pentagon);
            }

            // Hexagon
            function addHexagon(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const sides = 6;
                const radius = Math.min(scaledArea.width, scaledArea.height) / 4;
                const angle = 2 * Math.PI / sides;
                const path = [];

                for (let i = 0; i < sides; i++) {
                    path.push({
                        x: radius + radius * Math.cos(i * angle),
                        y: radius + radius * Math.sin(i * angle)
                    });
                }

                const hexagon = new fabric.Polygon(path, {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Hexagon"
                });

                restrictObject(hexagon);
                canvas.add(hexagon).setActiveObject(hexagon);
            }

            // Cloud
            function addCloud(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const area = getScaledArea(canvas, editableAreaData);

                const pathData = `
                    M60,40
                    C60,30 70,20 80,25
                    C85,10 105,10 110,25
                    C120,20 130,30 125,40
                    C135,40 140,55 130,60
                    C135,75 120,85 105,80
                    C95,90 75,90 65,80
                    C50,85 35,75 40,60
                    C30,55 35,40 50,40
                    Z
                `;

                const cloud = new fabric.Path(pathData, {
                    left: area.left + area.width / 2,
                    top: area.top + area.height / 2,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    scaleX: 0.8,
                    scaleY: 0.8,
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Cloud"
                });

                restrictObject(cloud);
                canvas.add(cloud).setActiveObject(cloud);
            }

            // Arrow
            function addArrow(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const area = getScaledArea(canvas, editableAreaData);

                const arrow = new fabric.Path('M10 80 L90 80 L90 60 L120 100 L90 140 L90 120 L10 120 Z', {
                    left: area.left + area.width / 2,
                    top: area.top + area.height / 2,
                    originX: 'left',
                    originY: 'top',
                    fill: color,
                    lockRotation: false,
                    lockScalingFlip: true,
                    hasControls: true,
                    shapeName: "Arrow"
                });

                restrictObject(arrow);
                canvas.add(arrow).setActiveObject(arrow);
            }

            // Speech Bubble
            function addSpeechBubble(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;
                const area = getScaledArea(canvas, editableAreaData);

                const bubble = new fabric.Path(
                    'M20 20 h160 a20 20 0 0 1 20 20 v60 a20 20 0 0 1 -20 20 h-100 l-40 30 v-30 h-20 a20 20 0 0 1 -20 -20 v-60 a20 20 0 0 1 20 -20 Z', {
                        left: area.left + area.width / 2,
                        top: area.top + area.height / 2,
                        originX: 'left',
                        originY: 'top',
                        fill: color,
                        lockRotation: false,
                        lockScalingFlip: true,
                        hasControls: true,
                        shapeName: "Speech Bubble"
                    });

                restrictObject(bubble);
                canvas.add(bubble).setActiveObject(bubble);
            }

            // Moon
            function addMoon(canvas, editableAreaData, restrictObject, container) {
                const color = container.find('.shape-color-picker')[0].value;

                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const imgScale = bgImg.scaleX;
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                const scaledArea = {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };

                const padding = 10;
                const size = Math.min(scaledArea.width, scaledArea.height) * 0.3;

                const bigCircle = new fabric.Circle({
                    radius: size / 2,
                    fill: color,
                    originX: 'center',
                    originY: 'center',
                    left: 0,
                    top: 0,
                    selectable: false
                });

                const smallCircle = new fabric.Circle({
                    radius: size / 2.2,
                    fill: 'black',
                    originX: 'center',
                    originY: 'center',
                    left: size / 5,
                    top: 0,
                    selectable: false,
                    globalCompositeOperation: 'destination-out'
                });

                const moonGroup = new fabric.Group([bigCircle, smallCircle], {
                    left: scaledArea.left + padding,
                    top: scaledArea.top + padding,
                    originX: 'left',
                    originY: 'top',
                    selectable: true,
                    hasControls: true,
                    lockRotation: false,
                    lockScalingFlip: true,
                    shapeName: "Moon"
                });

                restrictObject(moonGroup);
                canvas.add(moonGroup).setActiveObject(moonGroup);
            }


            // Utility: get scaled editable area based on background image
            function getScaledArea(canvas, editableAreaData) {
                const bgImg = canvas.backgroundImage;
                if (!bgImg) return editableAreaData; // fallback raw

                const imgScale = bgImg.scaleX; // assume uniform scaling
                const imgLeft = bgImg.left;
                const imgTop = bgImg.top;

                return {
                    left: imgLeft + editableAreaData.left * imgScale,
                    top: imgTop + editableAreaData.top * imgScale,
                    width: editableAreaData.width * imgScale,
                    height: editableAreaData.height * imgScale
                };
            }



            let hoveredCanvasInfo = null; // { canvas, container }

            document.addEventListener('keydown', (e) => {
                if (e.code === 'Space') {
                    if (hoveredCanvasInfo) {
                        const {
                            canvas
                        } = hoveredCanvasInfo;
                        const activeObj = canvas.getActiveObject();

                        // If editing text, don't block space
                        if (
                            activeObj &&
                            activeObj.isEditing &&
                            (activeObj.type === 'textbox' || activeObj.type === 'i-text' || activeObj.type ===
                                'text')
                        ) {
                            return; // let Fabric handle space for text input
                        }

                        // Otherwise prevent default and toggle move mode
                        e.preventDefault();

                        if (!canvas.isMoveMode) {
                            toggleMove(canvas, hoveredCanvasInfo.container);
                        }
                    }
                }
            });



            document.addEventListener('keyup', (e) => {
                if (e.code === 'Space' && hoveredCanvasInfo) {
                    const {
                        canvas,
                        container
                    } = hoveredCanvasInfo;

                    // Only disable if currently in move mode
                    if (canvas.isMoveMode) {
                        toggleMove(canvas, container);
                    }
                }
            });


            function toggleFullScreen(elem, canvas, container, editableAreaData) {
                const el = elem[0] || elem;

                function onFullScreenChange() {
                    const fsElement = document.fullscreenElement || document.webkitFullscreenElement ||
                        document.mozFullScreenElement || document.msFullscreenElement;

                    if (fsElement) {
                        // Fullscreen enabled
                        resizeCanvasToContainer(canvas, container);
                        fitScreen(canvas, container);
                    } else {
                        // Fullscreen exited - revert to original size or do same resize
                        resizeCanvasToContainer(canvas, container);
                        fitScreen(canvas, container);
                    }
                }

                // Listen fullscreen change event once
                document.addEventListener('fullscreenchange', onFullScreenChange);
                document.addEventListener('webkitfullscreenchange', onFullScreenChange);
                document.addEventListener('mozfullscreenchange', onFullScreenChange);
                document.addEventListener('MSFullscreenChange', onFullScreenChange);

                if (
                    document.fullscreenElement ||
                    document.webkitFullscreenElement ||
                    document.mozFullScreenElement ||
                    document.msFullscreenElement
                ) {
                    $('.full-screen__btn').find('i').addClass('la-expand-arrows-alt').removeClass(
                        'la-compress-arrows-alt');
                    if (document.exitFullscreen) document.exitFullscreen();
                    else if (document.webkitExitFullscreen) document.webkitExitFullscreen();
                    else if (document.mozCancelFullScreen) document.mozCancelFullScreen();
                    else if (document.msExitFullscreen) document.msExitFullscreen();
                } else {
                    $('.full-screen__btn').find('i').removeClass('la-expand-arrows-alt').addClass(
                        'la-compress-arrows-alt');
                    if (el.requestFullscreen) el.requestFullscreen();
                    else if (el.webkitRequestFullscreen) el.webkitRequestFullscreen();
                    else if (el.mozRequestFullScreen) el.mozRequestFullScreen();
                    else if (el.msRequestFullscreen) el.msRequestFullscreen();
                }
            }


            // Helper function to resize fabric canvas to container's current size
            function resizeCanvasToContainer(canvas, container) {
                const containerEl = container.find('.canvas-layout')[0];
                const width = containerEl.clientWidth;
                const height = containerEl.clientHeight;


                canvas.setWidth(width);
                canvas.setHeight(height);
                canvas.calcOffset();
                canvas.requestRenderAll();
            }

            function resizeCanvas(canvas, container) {
                const canvasEl = canvas.getElement(); // HTML canvas element
                const newHeight = container.find('.canvas-layout').height();
                const newWidth = container.find('.canvas-layout').width();

                // Resize container and upper canvas
                container.find('.canvas-container').css({
                    height: newHeight + 'px',
                    width: newWidth + 'px'
                });
                $(canvas.upperCanvasEl).css({
                    width: '100%',
                    height: '100%'
                });

                canvasEl.style.width = '100%';
                canvasEl.style.height = '100%';

                // Update Fabric internal dimensions
                canvas.setWidth(newWidth);
                canvas.setHeight(newHeight);

                canvas.calcOffset();
                canvas.requestRenderAll();
            }

            // Call resize on window resize
            $(window).on('resize', function() {
                $('.printSideAreaItem').each(function() {
                    const canvas = $(this).find('canvas')[0].fabricCanvas;
                    if (canvas) resizeCanvas(canvas, $(this));
                });
            });





            function enableMouseWheelZoom(canvas, container) {
                const canvasContainer = container.find('.canvas-container')[0];
                const zoomResetText = container.find('.zoom-reset');

                // Existing mouse wheel support
                canvasContainer.addEventListener('wheel', function(opt) {
                    opt.preventDefault();

                    let zoom = canvas.getZoom();
                    const zoomStep = 0.1;

                    if (opt.deltaY < 0) {
                        zoom *= (1 + zoomStep);
                    } else {
                        zoom *= (1 - zoomStep);
                    }

                    zoom = Math.min(Math.max(zoom, 0.1), 10);
                    canvas.zoomToPoint(canvas.getPointer(opt), zoom);
                    canvas.requestRenderAll();

                    const percentage = Math.round(zoom * 100) + '%';
                    zoomResetText.text(percentage);
                });

                // ADD: Touch pinch zoom support (scoped to this canvas)
                let initialDistance = 0;
                let initialZoom = 1;
                let isPinching = false;

                canvasContainer.addEventListener('touchstart', function(e) {
                    if (e.touches.length === 2) {
                        isPinching = true;
                        initialZoom = canvas.getZoom();

                        const touch1 = e.touches[0];
                        const touch2 = e.touches[1];
                        initialDistance = Math.sqrt(
                            Math.pow(touch2.clientX - touch1.clientX, 2) +
                            Math.pow(touch2.clientY - touch1.clientY, 2)
                        );
                        e.preventDefault();
                    }
                });

                canvasContainer.addEventListener('touchmove', function(e) {
                    if (isPinching && e.touches.length === 2) {
                        const touch1 = e.touches[0];
                        const touch2 = e.touches[1];
                        const currentDistance = Math.sqrt(
                            Math.pow(touch2.clientX - touch1.clientX, 2) +
                            Math.pow(touch2.clientY - touch1.clientY, 2)
                        );

                        const scale = currentDistance / initialDistance;
                        let newZoom = initialZoom * scale;
                        newZoom = Math.min(Math.max(newZoom, 0.1), 10);

                        // Get center point of pinch gesture
                        const centerX = (touch1.clientX + touch2.clientX) / 2;
                        const centerY = (touch1.clientY + touch2.clientY) / 2;
                        const rect = canvasContainer.getBoundingClientRect();
                        const pointer = {
                            x: centerX - rect.left,
                            y: centerY - rect.top
                        };

                        canvas.zoomToPoint(pointer, newZoom);
                        canvas.requestRenderAll();

                        const percentage = Math.round(newZoom * 100) + '%';
                        zoomResetText.text(percentage);

                        e.preventDefault();
                    }
                });

                canvasContainer.addEventListener('touchend', function(e) {
                    if (e.touches.length < 2) {
                        isPinching = false;
                    }
                });
            }

            // Assuming canvasMap is a global Map storing all canvases & containers:
            function updateAllCanvasSizes() {
                canvasMap.forEach(({
                    canvas,
                    container
                }) => {
                    const canvasEl = container.find('canvas')[0];
                    const newHeight = container.find('.canvas-layout').height();

                    // Update .canvas-container size
                    container.find('.canvas-container').css({
                        width: '100%',
                        height: newHeight + 'px'
                    });

                    // Update overlay canvas size
                    container.find('.upper-canvas').css({
                        width: '100%',
                        height: newHeight + 'px'
                    });

                    canvasEl.style.width = '100%';
                    canvasEl.style.height = newHeight + 'px';

                    canvas.setHeight(newHeight);
                    canvas.renderAll();
                });
            }


            // Call once on init
            updateAllCanvasSizes();

            // Bind to window resize
            $(window).on('resize', updateAllCanvasSizes);


            function fitScreen(canvas, container) {
                const editableAreaData = JSON.parse(canvas.lowerCanvasEl.dataset.selected || 'null');
                if (!editableAreaData) return;

                const bgImg = canvas.backgroundImage;
                if (!bgImg) return;

                const canvasWidth = canvas.getWidth();
                const canvasHeight = canvas.getHeight();
                const padding = 50;

                // Scaled editable area (image scale + offset)
                const areaLeft = bgImg.left + editableAreaData.left * bgImg.scaleX;
                const areaTop = bgImg.top + editableAreaData.top * bgImg.scaleY;
                const areaWidth = editableAreaData.width * bgImg.scaleX;
                const areaHeight = editableAreaData.height * bgImg.scaleY;

                // Zoom to fit editable area inside canvas
                const zoomX = (canvasWidth - padding * 2) / areaWidth;
                const zoomY = (canvasHeight - padding * 2) / areaHeight;
                let zoom = Math.min(zoomX, zoomY);
                zoom = Math.max(0.1, Math.min(zoom, 5));

                // Center editable area
                const areaCenterX = areaLeft + areaWidth / 2;
                const areaCenterY = areaTop + areaHeight / 2;

                const offsetX = canvasWidth / 2 - areaCenterX * zoom;
                const offsetY = canvasHeight / 2 - areaCenterY * zoom;

                canvas.viewportTransform = [zoom, 0, 0, zoom, offsetX, offsetY];
                canvas.requestRenderAll();

                const percentage = Math.round(zoom * 100) + '%';
                container.find('.zoom-reset').text(percentage);
            }

            // drawing js
            const drawingAreaModal = $('#drawingAreaModal');

            function enableDrawModal(canvas, container, editableAreaData, restrictObject) {
                drawingAreaModal.find('.modal-title').text(`@lang('Drawing area size for ') ${container.data('print-area-name')}`)
                drawingAreaModal.modal('show');
                currentDesingPrintAreaCanvas = canvas;
                currentDesingPrintAreaContainer = container;
                currentDesingPrintAreaEditableAreaData = editableAreaData;
                currentDesingPrintAreaRestrictObject = restrictObject;
            }


            // add quantity to the cart
            $('.addToCart').on('click', function() {
                let qtyInput = $('[name="quantity"]');

                if (qtyInput.val() < 1) {
                    qtyInput.focus();
                    return;
                }

                let formData = $('#printArea').serialize();
                $.ajax({
                    type: "post",
                    url: "{{ route('product.add.to.cart') }}",
                    data: formData,
                    success: function(response) {
                        if (response.status == "error") {
                            notify('error', response.message);
                            return;
                        }
                        notify('success', response.message);
                        $('#totalCartItems').text(response.data.totalProduct);
                    }
                });

            });

            const desingForm = $('#printArea');
            desingForm.on('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault(); // Prevent form submission
                    return false;
                }
            });

            $('.toggle-swap').on('click', function() {
                let alreadyOpened = $(this).closest('.left-layout__item').hasClass('active');
                $('.left-layout__item').removeClass('active');
                if (!alreadyOpened) $(this).closest('.left-layout__item').addClass('active');
            });

            $('.sidebar-layer__close').on('click', function() {
                $('.left-layout__item').removeClass('active');
            });

            $('.enableInstructionModal').on('click', function() {
                $('#instructionModal').modal('show');
            });

            function openDrawer(type) {
                $(`.drawer .left-layout__item:not(.${type})`).removeClass('active');
                $(`.drawer .left-layout__item.${type}:not('active')`).addClass('active');
            }

            desingForm.on('submit', function(e) {
                let allEmpty = true;
                $('input[name="selected_area[]"]').each(function() {
                    try {
                        const data = JSON.parse($(this).val());
                        if (data.objects && data.objects.length > 0) {
                            allEmpty = false; // At least one design exists
                            return false; // stop loop, we found a design
                        }
                    } catch (err) {}
                });

                if (allEmpty) {
                    e.preventDefault();
                    notify('error', `@lang('Please add your design before submitting. Let your creativity shine!')`);
                }
            });
        })(jQuery);
    </script>
@endpush
