<?php

namespace App\Services;

use App\Models\Catalog;
use App\Models\Product;
use App\Constants\Status;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;
use Illuminate\Support\Facades\Validator;
use App\Rules\SalePriceGreaterThanRegularPrice;

class ProductValidationService {

    /**
     * Validate product data.
     *
     * @param $request The request containing product data.
     * @param $product The product being validated.
     *
     * @return mixed The validation result.
     */

    public function validateProduct(Request $request, Product $product) {

        $validationRule = $this->validationRules($request, $product);
        $validationRule['type'] = 'required|string|in:general,description,seo,media,variants,print';

        return Validator::make($request->all(), $validationRule, $this->customValidationMessages());
    }
    /**
     * Define validation rules for product
     *
     * @return array The validation rules.
     */
    public function validationRules($request, $product = null) {
        $method = $request->type;

        if (method_exists($this, $method)) {
            return $this->{$method}($request, $product) ?? [];
        }

        return [];
    }

    private function general($request, $product) {
        $productTypes = implode(',', [Status::PRODUCT_TYPE_SIMPLE, Status::PRODUCT_TYPE_VARIABLE]);
        return [
            'name'                 => 'required|string',
            "catalog_id"           => [
                'required',
                function ($attribute, $value, $fail) {
                    $exists = Catalog::where('id', $value)
                        ->where('status', Status::ENABLE)
                        ->whereHas('catalogCategory', function ($query) {
                            $query->where('status', Status::ENABLE);
                        })
                        ->exists();

                    if (! $exists) {
                        $fail('The selected catalog is invalid or inactive.');
                    }
                }
            ],
            'product_type'         => 'required|in:' . $productTypes,
            'regular_price'        => 'nullable|required_if:product_type,' . Status::PRODUCT_TYPE_SIMPLE . '|numeric|gte:0',
            'sale_price'           => 'nullable|required_if:product_type,' . Status::PRODUCT_TYPE_SIMPLE . '|numeric|gte:0',
            'product_attributes'   => 'nullable|required_if:product_type,' . Status::PRODUCT_TYPE_VARIABLE . '|array|min:1',
            'product_attributes.*' => 'required_with:product_attributes|exists:attributes,id',
            'attribute_values'     => 'nullable|required_with:product_attributes|array|size:' . count($request['product_attributes'] ?? []),
            'attribute_values.*'   => 'required_with:attribute_values|exists:attribute_values,id',
        ];
    }

    private function description($request, $product) {
        return [
            'description'        => ['required', 'string', function ($attribute, $value, $fail) {
                if (trim(strip_tags($value)) === '') {
                    $fail('The ' . str_replace('_', ' ', $attribute) . ' field is required.');
                }
            }],
            'short_description' => 'required|array',
            'short_description.*' => 'required|string',
            'design_instruction' => 'nullable|array',
            'design_instruction.*' => 'nullable|string',
        ];
    }

    private function seo($request, $product) {
        return [
            'meta_title'       => 'nullable|string|max:255',
            'meta_keywords'    => 'nullable|array',
            'meta_description' => 'nullable|string',
        ];
    }

    private function media($request, $product) {
        $validation = [
            'main_image' => ['nullable', 'image', new FileTypeValidate(['jpeg', 'jpg', 'png']), 'max:2048'],
            'images'     => 'nullable|array|max:20',
        ];
        if ($product->product_type == Status::PRODUCT_TYPE_VARIABLE) {
            $validation['images.*']   = ['nullable', 'array'];
            $validation['images.*.*'] = ['nullable', 'image', new FileTypeValidate(['jpeg', 'jpg', 'png']), 'max:2048'];
        } else {
            $validation['images.*'] = ['nullable', 'image', new FileTypeValidate(['jpeg', 'jpg', 'png']), 'max:2048'];
        }
        return $validation;
    }

    private function variants($request, $product) {
        $variants = $product->productVariants;

        return [
            'variants_id'     => 'nullable|array|min:1',
            'variants_id.*'   => 'required|in:' . implode(',', $variants->pluck('id')->toArray()),
            'regular_price'   => 'required|array',
            'regular_price.*' => 'required|numeric|gte:0',
            'sale_price'      => 'required|array',
            'sale_price.*'    => ['required', 'numeric', 'gte:0', new SalePriceGreaterThanRegularPrice()],
            'is_published'    => 'nullable|array',
            'is_published.*'  => 'nullable|in:1,0',
        ];
    }

    private function print($request, $product) {
        $validation = [
            'published'           => 'nullable|in:1,0',
            "id"                  => 'nullable|array',
            "id.*"                => 'nullable|exists:product_print_areas,id',
            "name"                => 'required|array',
            "name.*"              => 'required|string',
            "print_area_width"    => 'required|array',
            "print_area_width.*"  => 'required|numeric|gt:0',
            "print_area_height"   => 'required|array',
            "print_area_height.*" => 'required|numeric|gt:0',
            "selected_area"       => 'required|array',
            "selected_area.*"     => 'required|string',
            "print_area_image"            => 'nullable|array',
            "print_area_image.*"          => ['nullable', 'image', new FileTypeValidate(['jpeg', 'jpg', 'png', 'svg'])],
        ];
        return $validation;
    }

    /**
     * Define custom validation error messages.
     *
     * @return array An array of custom validation error messages.
     */
    private function customValidationMessages() {
        return [
            'product_attributes.required_if' => 'Product attributes is required if product type is variable',
        ];
    }
}
