<?php

namespace App\Lib;

use App\Constants\Status;
use App\Models\AdminNotification;
use App\Models\UserLogin;
use App\Models\VendorLogin;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Hash;
use Socialite;

class SocialLogin {
    private $provider;
    private $model;
    private $fromApi;
    private $type;

    public function __construct($provider, $model, $type = NULL, $fromApi = false) {
        $this->provider = $provider;
        $this->model = $model;
        $this->type = $type;
        $this->fromApi = $fromApi;
        $this->configuration();
    }

    public function redirectDriver() {
        return Socialite::driver($this->provider)->redirect();
    }

    private function configuration() {
        $provider      = $this->provider;
        $configuration = gs('socialite_credentials')->$provider;
        $provider    = $this->fromApi && $provider == 'linkedin' ? 'linkedin-openid' : $provider;

        Config::set('services.' . $provider, [
            'client_id'     => $configuration->client_id,
            'client_secret' => $configuration->client_secret,
            'redirect'      => $this->type == 'vendor' ? route('vendor.social.login.callback', $provider)  : route('user.social.login.callback', $provider),
        ]);
    }

    public function login() {
        $provider      = $this->provider;
        $provider    = $this->fromApi && $provider == 'linkedin' ? 'linkedin-openid' : $provider;
        $driver     = Socialite::driver($provider);
        if ($this->fromApi) {
            try {
                $user = (object)$driver->userFromToken(request()->token)->user;
            } catch (\Throwable $th) {
                throw new Exception('Something went wrong');
            }
        } else {
            $user = $driver->user();
        }

        if ($provider == 'linkedin-openid') {
            $user->id = $user->sub;
        }

        $userData = $this->model::where('provider_id', $user->id)->first();

        if (!$userData) {
            if (!gs('registration')) {
                throw new Exception('New account registration is currently disabled');
            }
            $emailExists = $this->model::where('email', isset($user->email) ? $user->email : '')->exists();
            if ($emailExists) {
                throw new Exception('Email already exists');
            }

            $userData = $this->createUser($user, $this->provider);
        }
        if ($this->fromApi) {
            $tokenResult = $userData->createToken('auth_token')->plainTextToken;
            $this->loginLog($userData);
            return [
                'user'         => $userData,
                'access_token' => $tokenResult,
                'token_type'   => 'Bearer',
            ];
        }

        if($this->type == 'vendor'){
                Auth::guard('vendor')->login($userData);
        }else{
            Auth::login($userData);
        }


        $this->loginLog($userData);
        $redirection = Intended::getRedirection();

        if ($this->type == 'vendor') {
            return $redirection ? $redirection : to_route('vendor.home');
        }

        return $redirection ? $redirection : to_route('user.home');
    }

    private function createUser($user, $provider) {
        $general  = gs();
        $password = getTrx(8);

        $firstName = null;
        $lastName = null;

        if (isset($user->first_name)) {
            $firstName = $user->first_name;
        }
        if (isset($user->last_name)) {
            $lastName = $user->last_name;
        }

        if ((!$firstName || !$lastName) && isset($user->name)) {
            $firstName = preg_replace('/\W\w+\s*(\W*)$/', '$1', $user->name);
            $pieces    = explode(' ', $user->name);
            $lastName  = array_pop($pieces);
        }


        $modelClass = $this->model;
        $newUser = new $modelClass();
        $newUser->provider_id = $user->id;
        $newUser->email = $user->email;

        $newUser->password = Hash::make($password);
        $newUser->firstname = $firstName;
        $newUser->lastname = $lastName;

        $newUser->status = Status::VERIFIED;

        if ($this->type == "vendor") {
            $newUser->kv =  $general->kv ? Status::NO : Status::YES;
        }

        $newUser->ev = Status::VERIFIED;
        $newUser->sv = gs('sv') ? Status::UNVERIFIED : Status::VERIFIED;
        $newUser->ts = Status::DISABLE;
        $newUser->tv = Status::VERIFIED;
        $newUser->provider = $provider;
        $newUser->save();

        $adminNotification = new AdminNotification();
        if ($this->type == 'vendor') {
            $adminNotification->vendor_id = $newUser->id;
            $adminNotification->click_url = urlPath('admin.vendors.detail', $newUser->id);
            $adminNotification->title = 'New vendor registered';
        } else {
            $adminNotification->user_id = $newUser->id;
            $adminNotification->click_url = urlPath('admin.users.detail', $newUser->id);
            $adminNotification->title = 'New customer registered';
        }

        $adminNotification->save();
        $user = $this->model::find($newUser->id);
        return $user;
    }

    private function loginLog($user) {
        $ip = getRealIP();
        if ($this->type == 'vendor') {
            $exist = VendorLogin::where('vendor_ip', $ip)->first();
            $userLogin = new VendorLogin();
            $userLogin->vendor_id = $user->id;
            $userLogin->vendor_ip =  $ip;
        } else {
            $exist = UserLogin::where('user_ip', $ip)->first();
            $userLogin = new UserLogin();
            $userLogin->user_id = $user->id;
            $userLogin->user_ip =  $ip;
        }

        //Check exist or not
        if ($exist) {
            $userLogin->longitude =  $exist->longitude;
            $userLogin->latitude =  $exist->latitude;
            $userLogin->city =  $exist->city;
            $userLogin->country_code = $exist->country_code;
            $userLogin->country =  $exist->country;
        } else {
            $info = json_decode(json_encode(getIpInfo()), true);
            $userLogin->longitude =  isset($info['long']) ? implode(',', $info['long']) : '';
            $userLogin->latitude =  isset($info['lat']) ? implode(',', $info['lat']) : '';
            $userLogin->city =  isset($info['city']) ? implode(',', $info['city']) : '';
            $userLogin->country_code = isset($info['code']) ? implode(',', $info['code']) : '';
            $userLogin->country =  isset($info['country']) ? implode(',', $info['country']) : '';
        }

        $userAgent = osBrowser();
        $userLogin->browser = isset($userAgent['browser']) ? $userAgent['browser'] : '';
        $userLogin->os = isset($userAgent['os_platform']) ? $userAgent['os_platform'] : '';
        $userLogin->save();
    }
}
