!(function ($) {
    'use strict';
    // Define the fileUploader plugin
    $.fn.fileUploader = function (options) {
      // Reference to the current jQuery object (file uploader container)
      const uploader = this;
  
      // DataTransfer object for handling file drag-and-drop operations
      const dataTransfer = new DataTransfer();
  
      // Reference to the file input field within the uploader container
      let inputField;
  
      // Default settings for the file uploader
      const defaults = {
        filesName: 'files', // Name of the input field for files.
        preloaded: [], // Array to store preloaded files
        preloadedInputName: 'old', // Name for the input field that will store preloaded files.
        label: 'Drag & Drop files here or click to browse', // Default text displayed on the uploader container
        extensions: ['.jpg', '.jpeg', '.png'], // Accepted file extensions.
        mimes: ['image/jpeg', 'image/png', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'application/pdf', 'text/plain', 'audio/mpeg', 'audio/wav', 'audio/ogg', 'video/mp4', 'video/webm'], // Accepted MIME types.
        allowDuplicateFile: false, // Configuration for allowing duplicate files.
        maxSize: undefined, // Maximum file size allowed.
        maxFiles: undefined, // Maximum number of files allowed.
      };
  
      // Merge default settings with user-defined settings (if any)
      const settings = $.extend({}, defaults, options);
  
      // Initialize the file uploader
      const init = () => {
        uploader.each(function (i, element) {
          $(element).empty();
          // Create the uploader container
          const container = createUploaderContainer();
  
          // Append the container to the DOM element
          $(element).append(container);
  
          // Bind drag events to the container for visual feedback
          container.on('dragover dragleave', handleDragEvents);
  
          // Handle the drop of files
          container.on('drop', handleFileChange);
  
          // If preloaded files exists
          if (settings.preloaded.length) {
            // Change the container body
            container.addClass('has-files');
  
            // Get the uploaded files container
            const uploadedContainer = container.find('.uploaded');
  
            // Get the file extension
            // Set preloaded files preview
            settings.preloaded.forEach((file) => {
              uploadedContainer.append(createUploadedFilePreview(file.src, file.src.split('.').pop().toLowerCase(), file.id, true));
            });
          }
        });
      };
  
      // Handle drag events to change the appearance of the container
      const handleDragEvents = function (event) {
        haltEvent(event);
        $(this).toggleClass('drag-over', event.type === 'dragover');
      };
  
      const fileNames = () => Array.from(dataTransfer.files).map((file) => file.name);
  
      // Handle the change in files either through drag and drop or file selection
      const handleFileChange = function (event) {
        haltEvent(event);
        const input = $(this);
  
        // Retrieve the files either from file input or drag and drop
        // event.target.files: This retrieves the files selected through the <input type="file"> element.
        // event.originalEvent.dataTransfer.files: This retrieves the files that were dragged and dropped onto the element.
        const files = Array.from(event.target.files || event.originalEvent.dataTransfer.files);
  
        const validFiles = files.filter((file) => filterFiles(file));

        if(settings.maxFiles && !checkMaxFiles(files)) {
          return false;
        }
  
        if (validFiles.length) {
          input.removeClass('drag-over');
          updateUploader(input, validFiles);
        } else {
          input.prop('files', new DataTransfer().files);
        }
        settings.onSelect && settings.onSelect(validFiles);
      };
  
      const checkDuplicateFiles = (file) => {
        if (fileNames().includes(file.name)) {
          alert(`${file.name} is already added`);
          return false;
        }
        return true;
      };
  
      // Filter files based on settings
      const filterFiles = (file) => {
        return (settings.allowDuplicateFile || checkDuplicateFiles(file)) && (!settings.extensions || checkExtension(file)) && (!settings.mimes || checkMime(file)) && (!settings.maxSize || checkSize(file));
      };
  
      // Generate a unique ID
      const generateId = () => Date.now() + Math.floor(100 * Math.random() + 1);
  
      // Create the uploader container
      const createUploaderContainer = () => {
        const container = $('<div>', { class: 'file-uploader' });
  
        // Create file input
        inputField = $('<input>', {
          type: 'file',
          id: settings.filesName + '-' + generateId(),
          name: settings.filesName + '[]',
          accept: settings.extensions.join(','),
          multiple: '',
        }).appendTo(container);
  
        // Create container for uploaded files
        $('<div>', { class: 'uploaded' }).appendTo(container);
  
        // Create upload text
        let uploadText = $('<div>', { class: 'upload-text' }).appendTo(container);
        $('<i>', { class: 'iui-cloud-upload' }).appendTo(uploadText);
        $('<span>', { text: settings.label }).appendTo(uploadText);
  
        // Trigger file input click on container click
        container.on('click', (e) => {
          haltEvent(e);
          inputField.trigger('click');
        });
  
        // Stop click event propagation for input
        inputField.on('click', (e) => e.stopPropagation());
  
        // Bind the event handler on file select
        container.on('change', inputField, handleFileChange);
        return container;
      };
  
      // Prevent default event action and stop propagation
      const haltEvent = function (event) {
        event.preventDefault();
        event.stopPropagation();
      };
  
      // Create a preview container for the uploaded file
      let createUploadedFilePreview = (src, extension, id, preloaded) => {
        let visualElement;
        // Create the uploaded file container
        const uploadedFile = $('<div>', { class: 'uploaded-file', 'data-index': id });
  
        // Create the file tag for preview
        $('<img>', { src: src }).appendTo(uploadedFile);
  
        // Create the delete button
        const deleteButton = $('<button>', { class: 'delete-file-button' }).appendTo(uploadedFile);
  
        // Create the icon for delete button
        $('<i>', { class: 'iui-close' }).appendTo(deleteButton);
  
        if (preloaded) {
          uploadedFile.attr('data-preloaded', true);
          $('<input>', {
            type: 'hidden',
            name: `${settings.preloadedInputName}[]`,
            value: id,
          }).appendTo(uploadedFile);
        } else {
          uploadedFile.attr('data-index', id);
        }
  
        // Prevent click event propagation
        uploadedFile.on('click', (e) => haltEvent(e));
  
        uploadedFile.find('.delete-file-button').on('click', deleteFileClickHandler);
        return uploadedFile;
      };
  
      // Handle delete file button click
      const deleteFileClickHandler = function (event) {
        haltEvent(event);
        const uploadedFile = $(this).closest('.uploaded-file');
        const container = uploadedFile.parent();
  
        // Handle deletion of preloaded files
        if (uploadedFile.data('preloaded')) settings.preloaded = settings.preloaded.filter((item) => item.id !== uploadedFile.data('index'));
        else {
          const index = parseInt(uploadedFile.data('index'));
  
          // Update indices for subsequent files
          container.find('.uploaded-file[data-index]').each(function (i, element) {
            if (i > index) {
              $(element).attr('data-index', i - 1);
            }
          });
  
          // Remove file from dataTransfer
          dataTransfer.items.remove(index);
  
          // Update input field with remaining files
          inputField.prop('files', dataTransfer.files);
        }
  
        // Remove the file from the container
        uploadedFile.remove();
  
        // Remove "has-files" class if no files left
        if (!container.children().length) {
          container.parent().removeClass('has-files');
          settings.onRemove() ?? settings.onRemove();
        }
      };
  
      // Validate file extension
      const checkExtension = (file) => {
        if (settings.extensions.includes('*')) {
          return true;
        }
  
        if (settings.extensions.indexOf(file.name.replace(new RegExp('^.*\\.'), '.')) < 0) {
          alert(`The file "${file.name}" does not match with the accepted file extensions: "${settings.extensions.join('", "')}"`);
  
          return false;
        }
  
        return true;
      };
  
      // Validate MIME type
      const checkMime = (file) => {
        if (settings.mimes.includes('*')) {
          return true;
        }
  
        if (settings.mimes.indexOf(file.type) < 0) {
          alert(`The file "${file.name}" does not match with the accepted mime types: "${settings.mimes.join('", "')}"`);
          return false;
        }
  
        return true;
      };
  
      // Validate file size
      let checkSize = (file) => {
        if (file.size > settings.maxSize) {
          alert(`The file "${file.name}" exceeds the maximum size of ${settings.maxSize / 1024 / 1024}Mb`);
          return false;
        }
  
        return true;
      };
  
      // Validate maximum number of files
      const checkMaxFiles = (files) => {
        const totalFiles = files.length + dataTransfer.items.length + settings.preloaded.length;
        
        if (totalFiles >= settings.maxFiles) {
          alert(`You can only upload up to ${settings.maxFiles} files.`);
          return false;
        }
        return true;
      };
  
      // Update the uploader with the valid files
      let updateUploader = (input, files) => {
        input.addClass('has-files');
        const uploadedContainer = input.find('.uploaded');
        const fileInput = input.find('input[type="file"]');
  
        files.forEach((file) => {
          dataTransfer.items.add(file);
  
          uploadedContainer.append(createUploadedFilePreview(URL.createObjectURL(file), file.type, dataTransfer.items.length - 1), false);
        });
  
        fileInput.prop('files', dataTransfer.files);
      };
  
      init();
      return this;
    };
  })(jQuery);
  